#include <stdio.h>
#include <openbabel/mol.h>
#include <openbabel/obconversion.h>
#include <iostream>
#include <set>
#include <math.h>

using namespace std;
using namespace OpenBabel;

//------------------------------------------------------------------------------

// function prototypes
bool ReadStructure(OBMol& mol, const string& name);
bool WriteStructure(OBMol& mol, const string& name);
void AlignStructure(OBMol& mol);
void CutStructure(OBMol& imol, OBMol& omol, double coneAngle, double coneRadius);
void CleanStructure(OBMol& mol);
int CountFreeOxygens(OBMol& mol);
int CountOxygens(OBMol& mol);

//------------------------------------------------------------------------------

int main(int argc,char* argv[])
{

    OBMol imol; // input molecule
    OBMol omol; // output molecule


    string iname = "test/output.xyz";
    if( ReadStructure(imol,iname) == false ) return(1);
    int freeOx = CountFreeOxygens(imol);

    cout << "Number of atoms: " << imol.NumAtoms() << endl;
    cout << "Number of oxygens: " << CountOxygens(imol) << endl;
    cout << "Number of silicons: " << imol.NumAtoms() - CountOxygens(imol) << endl;
    cout << "Number of free oxygens: " << freeOx << endl;

    return(0);
}

//------------------------------------------------------------------------------

// this will cut structure according to user setup coneAngle - in rad, coneRadius in A.
// cleanning is done by another method

void CutStructure(OBMol& imol, OBMol& omol, double coneAngle, double coneRadius)
{
    cout << endl;
    cout << "Cutting structure ..." << endl;

    // selected atoms
    // in pass 1 only silicons
    vector<unsigned int> selected_atoms;

    // first pass - select silicon atoms according to user criteria
    // babel indexes atoms from 1
    for(unsigned int i=1; i <= imol.NumAtoms(); i++){
        OBAtom* p_atom = imol.GetAtom(i);
        if( p_atom->GetAtomicNum() != 14 ) continue; // only silicons
        // position
        double x = p_atom->GetX();
        double y = p_atom->GetY();
        double z = p_atom->GetZ();


        double r = coneRadius-z*sin(coneAngle/2);

        if( r > sqrt(x*x+y*y) ){
            // select atom
            selected_atoms.push_back(i);
        }
    }

    // info
    cout << "   Number of selected silicons : " << selected_atoms.size() << endl;

    // second pass - select oxygen atoms that are connected to selected silicons

    // selected oxygen atoms will be stored as a set
    // http://www.cplusplus.com/reference/set/set/
    set<unsigned int> selected_oxygens;

    // vector items are indexed from zero
    for(size_t i=0; i < selected_atoms.size(); i++){
        // get selected atom
        OBAtom* p_atom = imol.GetAtom( selected_atoms[i] );

        // go through its neighbour atoms
        OBBondIterator    ni;

        OBAtom* p_natom = p_atom->BeginNbrAtom(ni);
        while( p_natom != NULL ){
            // this assumes that all neighbour atoms are oxygens
            // otherwise use a filter by Z

            // get atom index
            unsigned int oi = p_natom->GetIdx();

            // inser to a set, duplicities are automatically handled by a container
            selected_oxygens.insert(oi);

            // next atom
            p_natom = p_atom->NextNbrAtom(ni);
        }
    }

    // copy selected oxygens to silicons
    selected_atoms.insert(selected_atoms.end(),selected_oxygens.begin(),selected_oxygens.end());
    // create new molecule

    // reserve storage for atoms
    omol.ReserveAtoms(selected_atoms.size());

    // copy data
    for(size_t i=0; i < selected_atoms.size(); i++){
        // get selected atom
        OBAtom* p_satom = imol.GetAtom( selected_atoms[i] );

        // create new atom and copy relevant data
        OBAtom* p_tatom  = omol.NewAtom();
        p_tatom->SetAtomicNum(p_satom->GetAtomicNum()); // set Z
        p_tatom->SetVector(p_satom->GetVector()); //set coordinates
    }

    // create bonds
    omol.ConnectTheDots();
}

//------------------------------------------------------------------------------


void CleanStructure(OBMol& mol)
{
    cout << endl;
    cout << "Cleaning structure ..." << endl;

    bool nextCycleNeeded = false;
    // make list of silicon atoms that will be deleted
    vector<OBAtom*> atoms_to_be_deleted;
    // throught all atoms
    for( unsigned int i=1; i <= mol.NumAtoms(); i++ ){
        OBAtom* p_atom = mol.GetAtom(i);
        if( p_atom->GetAtomicNum() != 14 ) continue; // only silicons

        // count number of neighbour silicons
        // this needs to traverse neighbour atoms twice in two layers

        int num_of_nsilicons = 0;

        // go through first layer of neighbour atoms
        OBBondIterator    n1i;
        OBAtom* p_n1atom = p_atom->BeginNbrAtom(n1i);

        while( p_n1atom != NULL ){
            // this assumes that all neighbour atoms are oxygens

            // go through second layer of neighbour atoms
            OBBondIterator    n2i;
            OBAtom* p_n2atom = p_n1atom->BeginNbrAtom(n2i);
            while( p_n2atom != NULL ){
                if( p_n2atom != p_atom ){ // avoid self counting
                    // this assumes that all neighbour atoms are silicons
                    num_of_nsilicons++;
                }

                // next atom
                p_n2atom = p_n1atom->NextNbrAtom(n2i);
            }

            // next atom
            p_n1atom = p_atom->NextNbrAtom(n1i);
        }

        if( num_of_nsilicons <= 1 ){
            // mark atom for deletion
            // 0 - orphan SiO4
            // 1 - SiO4 bound by one -O- bridge to the rest of the structure
            atoms_to_be_deleted.push_back(p_atom);
        }

        //removes Si-O-Si-O- chains by recursion
        if( num_of_nsilicons == 1)
            nextCycleNeeded = true;

    }

    cout << "   Number of orphaned silicons : " << atoms_to_be_deleted.size() << endl;

    // delete silicon atoms from orphaned units
    for(size_t i=0; i < atoms_to_be_deleted.size(); i++){
        mol.DeleteAtom(atoms_to_be_deleted[i]);
    }

    // now delete all oxygen atoms that are orphans
    atoms_to_be_deleted.clear();

    // throught all atoms
    for( unsigned int i=1; i <= mol.NumAtoms(); i++ ){
        OBAtom* p_atom = mol.GetAtom(i);
        if( p_atom->GetAtomicNum() != 8 ) continue; // only oxygens
        if( p_atom->GetValence() == 0 ) atoms_to_be_deleted.push_back(p_atom); // no bond
    }

    cout << "   Number of orphaned oxygens : " << atoms_to_be_deleted.size() << endl;

    // delete orphaned oxygen atoms
    for(size_t i=0; i < atoms_to_be_deleted.size(); i++){
        mol.DeleteAtom(atoms_to_be_deleted[i]);
    }

    if(nextCycleNeeded)
    {
        cout << "   Detected Si-O chains, cleaning one more time ...";

        CleanStructure(mol);
    }
}

//------------------------------------------------------------------------------

// read structure in xyz format
// it solves obscure behaviour of standard openbabel xyz reader

bool ReadStructure(OBMol& mol, const string& name)
{
   ifstream ifs;

   // open file and test if it succeeded
   ifs.open(name.c_str());
   if( ! ifs ){
       cerr << ">>> ERROR: Unable to open input file : '" << name << "'!" << endl;
       return(false);
    }

   // read number of atoms
   int numofatoms = 0;
   ifs >> numofatoms;
   if( (! ifs) || (numofatoms <= 0) ){
       cerr << ">>> ERROR: Unable to read number of atoms or illegal number of atoms!" << endl;
       return(false);
    }

   // skip the rest of line and read comment line
   string buffer;
   getline(ifs,buffer);
   getline(ifs,buffer);

   // reserve storage for atoms
   mol.ReserveAtoms(numofatoms);


   // read atoms
   for(int i=0; i < numofatoms; i++){
       string symbol;
       double x,y,z;

       // read data
       ifs >> symbol >> x >> y >> z;


       if( ! ifs ){
           cerr << ">>> ERROR: Unable to read atom " << i + 1 << "!" << endl;
           return(false);
       }

       // create atom
       OBAtom* p_atom  = mol.NewAtom();
       int atomicNum = etab.GetAtomicNum(symbol.c_str());
       p_atom->SetAtomicNum(atomicNum); // set Z
       p_atom->SetVector(x,y,z); //set coordinates

   }


   // create bonds
   mol.ConnectTheDots();

   // OK, this causes troubles in standard xyz parser
   // mol.PerceiveBondOrders();

   return(true);
}
//------------------------------------------------------------------------------

// translates structure to the center of coord system, alignes x,y axis

void AlignStructure(OBMol &mol)
{

    cout << "   Aligning structure ..." << endl;
    double byMaxXcoord [2];
    double byMaxYcoord [2];
    double byMinXcoord [2];
    double byMinYcoord [2];
    double x, y;
    //initialize maxCoords and minCoords field to count translation vector and rotation matrix
    for (unsigned int i=1; i<=mol.NumAtoms();i++){
        x = mol.GetAtom(i)->GetVector().GetX();
        y = mol.GetAtom(i)->GetVector().GetY();

        if(i==0){
            byMaxXcoord[0] = x;
            byMaxYcoord[0] = x;
            byMaxXcoord[1] = y;
            byMaxYcoord[1] = y;
            byMinXcoord[0] = x;
            byMinYcoord[0] = x;
            byMinXcoord[1] = y;
            byMinYcoord[1] = y;

        }
        else
        {
            if(byMaxXcoord[0] < x){
                byMaxXcoord[0] = x;
                byMaxXcoord[1] = y;

            }
            if(byMaxYcoord[1] < y){
                byMaxYcoord[0] = x;
                byMaxYcoord[1] = y;

            }
            if(byMinXcoord[0] > x){
                byMinXcoord[0] = x;
                byMinXcoord[1] = y;

            }
            if(byMinYcoord[1] > y){
                byMinYcoord[0] = x;
                byMinYcoord[1] = y;

            }

        }
    }

    //declaring rotation matrix
    double alpha = atan(byMaxXcoord[1]/byMaxXcoord[0]);
    double beta  = atan(byMaxYcoord[1]/byMaxYcoord[0]);
    double fi = -(alpha+beta)/2;
    double M [3][3];
    M[0][0] = cos(fi); M[0][1] = sin(fi); M[0][2] = 0;
    M[1][0] =-sin(fi); M[1][1] = cos(fi); M[1][2] = 0;
    M[2][0] = 0;       M[2][1] = 0;       M[2][2] = 1;

    //declaring translation vector, assuming that z-axis is correct
    vector3 V(-(byMaxXcoord[0]+byMinXcoord[0])/2,-(byMaxYcoord[1]+byMinYcoord[1])/2,0);
    vector3 newV;

    cout << "   Rotating structure by "  << fi << " rad." << endl;
    //writing new coords to each atom
    for (unsigned int i=1; i<=mol.NumAtoms();i++){
        OBAtom* p_atom = mol.GetAtom(i);

        newV = p_atom->GetVector();
        newV+= V;
        newV*= M;
        p_atom->SetVector(newV);


    }


}


//------------------------------------------------------------------------------

// write structure in xyz format

bool WriteStructure(OBMol& mol, const string& name)
{
   ofstream ofs;

   // open file and test if it succeeded
   ofs.open(name.c_str());
   if( ! ofs ){
       cerr << ">>> ERROR: Unable to open output file : '" << name << "'!" << endl;
       return(false);
    }

   // setup standard converter
   OBConversion conv(NULL,&ofs);
   conv.SetOutFormat("XYZ");

   // save structure
   if( conv.Write(&mol) == false ){
       cerr << ">>> ERROR: Unable to save output file : '" << name << "'!" << endl;
       return(false);
   }

   return(true);
}

int CountFreeOxygens(OBMol& mol)
{
    int freeOxygens = 0;

    for (unsigned int i=1; i<=mol.NumAtoms();i++)
    {
        int num_of_nsilicons = 0;
        OBAtom* p_atom = mol.GetAtom(i);
        if(p_atom->GetAtomicNum() != 8) continue;

        OBBondIterator bi;

        OBAtom* p_siatom = p_atom->BeginNbrAtom(bi);
        while( p_siatom != NULL ){
            num_of_nsilicons++;

            // next atom
            p_siatom = p_atom->NextNbrAtom(bi);
        }

        if (num_of_nsilicons == 1)
            freeOxygens++;

    }

    return(freeOxygens);

}

int CountOxygens(OBMol& mol)
{
    int oxygens = 0;

    for (unsigned int i=1; i<=mol.NumAtoms();i++)
    {
        OBAtom* p_atom = mol.GetAtom(i);
        if(p_atom->GetAtomicNum() != 8) continue;
        oxygens++;

    }

    return(oxygens);
}

//------------------------------------------------------------------------------
